import traceback
import logging
import time
from ethoscope.web_utils.control_thread import ControlThread, ExperimentalInformation
from ethoscope.utils.description import DescribedObject
from ethoscope.hardware.input.cameras import V4L2Camera, OurPiCameraAsync
import tempfile
import shutil
import threading, queue
import glob
import datetime
import os

#from cv2 import VideoWriter, VideoWriter_fourcc, imwrite
import cv2

#streaming socket
import socket, struct, io
import pickle

STREAMING_PORT = 8887

 
class cameraCaptureThread(threading.Thread):
    '''
    This opens a PiCamera process for recording or streaming video
    For recording, files are saved in chunks of time duration
    In principle, the two activities couldbe done simultaneously ( see https://picamera.readthedocs.io/en/release-1.12/recipes2.html#capturing-images-whilst-recording )
    but for now they are handled independently
    '''
        
    _VIDEO_CHUNCK_DURATION = 30 * 10
    def __init__(self, cameraClass, camera_kwargs, video_prefix, video_root_dir, img_path, width, height, fps, bitrate, stream=False):
        self._img_path = img_path
        
        self._resolution = (width, height)
        self._fps = fps
        self._bitrate = bitrate
        self._video_prefix = video_prefix
        self._video_root_dir = video_root_dir
        self._stream = stream
        self.camera = cameraClass ( target_fps = fps , target_resolution = (width, height), **camera_kwargs )
        
        self.video_file_index = 0
        self.stop_camera_activity = False

        super(cameraCaptureThread, self).__init__()

    def _get_video_chunk_filename(self, ext='h264'):
        '''
        we save the files in chunks that will have to be merged togheter at a later point
        this names the next chunck
        '''

        self.video_file_index += 1
        w,h = self._resolution
        video_info= "%ix%i@%i" %(w, h, self.camera.fps)
        video_filename = '%s_%s_%05d.%s' % (self._video_prefix, video_info, self.video_file_index, ext)
        return video_filename

    def _save_preview_frame(self, frame, writing_status):
        '''
        '''

        writing_status = ['', 'Writing'][writing_status]
        timestamp = datetime.datetime.now().strftime("%m/%d/%Y, %H:%M:%S") + " FPS: " + str(round(self.camera.fps,2)) + " " + writing_status
        cv2.putText(frame, timestamp , (20,20) , 1 , 1 , (255,255,255) )
        cv2.imwrite(self._img_path, frame)
        self.preview_time = time.time()

    def run (self):
        '''
        Iterates the camera object for images and writes them the to a video file, dividing the video in multiple AVI
        Every 5 seconds, updates the preview frame served over the network by the webserver adding some info text on it
        '''

        if self._stream:
            server_socket = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            server_socket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
            server_socket.bind(('', STREAMING_PORT))
            server_socket.listen(5)
            logging.info("Socket stream initiliased.")

        # creates a destination folder for the video, if it does not exist
        try:
            video_dirname = os.path.dirname(self._video_prefix)
            if not os.path.exists ( video_dirname ): os.makedirs( video_dirname )
                
        except OSError as e:
            raise e

        self.start_time = time.time()
        self.preview_time = time.time()
        
        writer = None

        while not self.stop_camera_activity:

            if self._stream:
                logging.info("Waiting for a connection to start streaming")
                client_socket, client_address = server_socket.accept() # blocking call
                logging.info("Connection established!")
          
            for ix, (_, frame) in enumerate (self.camera):

                if self.stop_camera_activity: break
                
                if writer and writer.isOpened() and not self._stream:
                    writer.write(frame)
                
                # Wait for the first 150 frames before opening the video writer object
                # This is done to calcualate a decent approximation of actual FPS
                if time.time() - self.start_time >= self._VIDEO_CHUNCK_DURATION or ix == 150 and not self._stream:
                    if writer:
                        writer.release()
                    
                    writer = cv2.VideoWriter(self._get_video_chunk_filename(ext='avi'), cv2.VideoWriter_fourcc(*'mp4v'), self.camera.fps, (self.camera.width, self.camera.height))
                    if not writer.isOpened():
                        logging.error('Error: failed to open Video writer destination. The Video file cannot be saved.')

                    self.start_time = time.time()

                # annotates a preview frame every 5 seconds
                if (( time.time() - self.preview_time ) > 5) and not self._stream:
                    self._save_preview_frame(frame, writing_status = writer is not None)
                    
                if self._stream:

                    frame = cv2.resize(frame, (640,480))
                    frame = cv2.putText(frame, 'FPS: ' + str(round(self.camera.fps,2)) , (20,20) , 1 , 1 , (255,255,255) )
                    _, frame = cv2.imencode('.jpg', frame, [int(cv2.IMWRITE_JPEG_QUALITY), 90])

                    data = pickle.dumps(frame)
                    message = struct.pack("Q", len(data)) + data
                    client_socket.sendall(message)
            
        # out of the loop - exit signal received
        self.camera._close()

        if self._stream:
            client_socket.close()
            server_socket.close()

        if writer:
            writer.release()

class GeneralVideoRecorder(DescribedObject):

    _description  = {  "overview": "A video simple recorder",
                            "arguments": [
                                {"type": "number", "name":"width", "description": "The width of the frame","default":1280, "min":480, "max":1980,"step":1},
                                {"type": "number", "name":"height", "description": "The height of the frame","default":960, "min":360, "max":1080,"step":1},
                                {"type": "number", "name":"fps", "description": "The target number of frames per seconds","default":25, "min":1, "max":25,"step":1},
                                {"type": "number", "name":"bitrate", "description": "The target bitrate","default":200000, "min":0, "max":10000000,"step":1000}
                               ]}
    status = "recording" #this is the default status. The alternative is streaming

    def __init__(self, cameraClass, camera_kwargs, video_prefix, video_dir, img_path, width=1280, height=960, fps=25, bitrate=200000, stream = False):

        self._stream = stream

        #This used to be a process but it's best handled as a thread. See also commit https://github.com/gilestrolab/ethoscope/commit/c2e8a7f656611cc10379c8e93ff4205220c8807a
        self._p = cameraCaptureThread(cameraClass, camera_kwargs, video_prefix, video_dir, img_path, width, height, fps, bitrate, stream)

    def start_recording(self):
        '''
        '''
        self._p.start()

        #while self._p.is_alive():
        #    time.sleep(.25)
            
    def stop(self):
        '''
        '''
        logging.info("The control thread asked me to stop the camera")
        self._p.stop_camera_activity = True

        if self._stream: 
            try:
                self._p.connection.close()
            except:
                pass

        self._p.join(10)

class HDVideoRecorder(GeneralVideoRecorder):
    _description  = { "overview": "A preset 1920 x 1080, 25fps, bitrate = 5e5 video recorder. "
                                  "At this resolution, the field of view is only partial, "
                                  "so we effectively zoom in the middle of arenas","arguments": []}
    status = "recording"

    def __init__(self, cameraClass, camera_kwargs, video_prefix, video_dir, img_path):
        super(HDVideoRecorder, self).__init__(cameraClass, camera_kwargs, video_prefix, video_dir, img_path, width=1920, height=1080,fps=25,bitrate=1000000)


class StandardVideoRecorder(GeneralVideoRecorder):
    _description  = { "overview": "A preset 1280 x 960, 25fps, bitrate = 2e5 video recorder.", "arguments": []}
    status = "recording"
    
    def __init__(self, cameraClass, camera_kwargs, video_prefix, video_dir, img_path):
        super(StandardVideoRecorder, self).__init__(cameraClass, camera_kwargs, video_prefix, video_dir, img_path, width=1280, height=960,fps=25,bitrate=500000)

class Streamer(GeneralVideoRecorder):
    #hiding the description field will not pass this class information to the node UI
    _description  = { "overview": "A preset 640 x 480, 25fps, bitrate = 2e5 streamer. Active on port 8008.", "arguments": [], 'hidden': True}
    status = "streaming"
    
    def __init__(self, cameraClass, camera_kwargs, video_prefix, video_dir, img_path):
        super(Streamer, self).__init__(cameraClass, camera_kwargs, video_prefix, video_dir, img_path, width=640, height=480, fps=20, bitrate=500000, stream=True)
        

class ControlThreadVideoRecording(ControlThread):

    _evanescent = False
    _option_dict = {
            "camera": {
                    "possible_classes":[OurPiCameraAsync, V4L2Camera],
                    },
            "recorder": {
                    "possible_classes":[StandardVideoRecorder, HDVideoRecorder, GeneralVideoRecorder, Streamer],
                },
            "experimental_info":{
                    "possible_classes":[ExperimentalInformation],
                    }
                }
                
    for k in _option_dict:
        _option_dict[k]["class"] =_option_dict[k]["possible_classes"][0]
        _option_dict[k]["kwargs"] ={}


    _tmp_last_img_file = "last_img.jpg"
    _dbg_img_file = "dbg_img.png"
    _log_file = "ethoscope.log"
    
    _hidden_options = {'camera'}

    def __init__(self, machine_id, name, version, ethoscope_dir, data=None, *args, **kwargs):

        # for FPS computation
        self._last_info_t_stamp = 0
        self._last_info_frame_idx = 0
        
        # Metadata
        self._recorder = None
        self._machine_id = machine_id
        self._device_name = name
        self._video_root_dir = ethoscope_dir
        self._tmp_dir = tempfile.mkdtemp(prefix="ethoscope_")


        #todo add 'data' -> how monitor was started to metadata
        self._info = {"status": "stopped",
                        "time": time.time(),
                        "error": None,
                        "log_file": os.path.join(ethoscope_dir, self._log_file),
                        "dbg_img": os.path.join(ethoscope_dir, self._dbg_img_file),
                        "last_drawn_img": os.path.join(self._tmp_dir, self._tmp_last_img_file),
                        "id": machine_id,
                        "name": name,
                        "version": version,
                        "experimental_info": {}
                        }

        self._parse_user_options(data)
        super(ControlThread, self).__init__()

    def _update_info(self):
        if self._recorder is None:
            return
        self._last_info_t_stamp = time.time()


    def _parse_one_user_option(self, field, data):

        try:
            subdata = data[field]
        except KeyError:
            logging.warning("No field %s, using default" % field)
            return None, {}

        Class = eval(subdata["name"])
        kwargs = subdata["arguments"]

        return Class, kwargs


    def run(self):

        try:
            self._info["status"] = "initialising"
            logging.info("Starting Monitor thread")
            self._info["error"] = None


            self._last_info_t_stamp = 0
            self._last_info_frame_idx = 0

            ExpInfoClass = self._option_dict["experimental_info"]["class"]
            exp_info_kwargs = self._option_dict["experimental_info"]["kwargs"]
            self._info["experimental_info"] = ExpInfoClass(**exp_info_kwargs).info_dic
            self._info["time"] = time.time()

            date_time = datetime.datetime.fromtimestamp(self._info["time"])
            formatted_time = date_time.strftime('%Y-%m-%d_%H-%M-%S')

            try:
                code = self._info["experimental_info"]["code"]
            except KeyError:
                code = "NA"
                logging.warning("No code field in experimental info")

            file_prefix = "%s_%s_%s" % (formatted_time, self._machine_id, code)

            self._output_video_full_prefix = os.path.join ( self._video_root_dir, self._machine_id, self._device_name, formatted_time, file_prefix )

            RecorderClass = self._option_dict["recorder"]["class"]
            recorder_kwargs = self._option_dict["recorder"]["kwargs"] # {'width': 1280, 'height': 960, 'fps': 25, 'bitrate': 200000}

            cameraClass = self._option_dict["camera"]["class"]
            camera_kwargs = self._option_dict["camera"]["kwargs"]
            
            self._recorder = RecorderClass( cameraClass, camera_kwargs, 
                                            video_prefix = self._output_video_full_prefix,
                                            video_dir = self._video_root_dir,
                                            img_path = self._info["last_drawn_img"],
                                            **recorder_kwargs)

            self._info["status"] = self._recorder.status # "recording" or "streaming"
            logging.info( "Started %s" % self._recorder.status )
            
            self._recorder.start_recording()

        except Exception as e:
            self.stop(traceback.format_exc())

        #for testing purposes
        if self._evanescent:
            self.stop()
            os._exit(0)


    def stop(self, error=None):
        '''
        '''
        self._info["status"] = "stopping"
        self._info["time"] = time.time()
        self._info["experimental_info"] = {}

        if self._recorder is not None:
            logging.info("Control thread asking recorder to stop")
            self._recorder.stop()
            self._recorder = None

        self._info["status"] = "stopped"
        self._info["time"] = time.time()
        self._info["error"] = error

        if error is not None:
            logging.error("Recorder closed with an error:")
            logging.error(error)
        else:
            logging.info("Recorder closed all right")
