#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
Test runner for notification system tests.
Demonstrates the comprehensive test coverage for the enhanced notification system.
"""

import subprocess
import sys
import os
from pathlib import Path

def run_tests():
    """Run all notification system tests."""
    
    # Change to the node directory
    node_dir = Path(__file__).parent
    os.chdir(node_dir)
    
    print("🧪 Running Notification System Tests")
    print("=" * 50)
    
    # Test categories to run
    test_categories = [
        {
            'name': 'Unit Tests - Base Analyzer',
            'path': 'tests/unit/notifications/test_base.py',
            'description': 'Tests for device failure analysis, log retrieval, and user management'
        },
        {
            'name': 'Unit Tests - Email Service',
            'path': 'tests/unit/notifications/test_email.py',
            'description': 'Tests for email notifications, SMTP handling, and message formatting'
        },
        {
            'name': 'Integration Tests',
            'path': 'tests/integration/notifications/',
            'description': 'End-to-end workflow tests and component interaction tests'
        }
    ]
    
    total_passed = 0
    total_failed = 0
    
    for category in test_categories:
        print(f"\n📋 {category['name']}")
        print(f"   {category['description']}")
        print("-" * 50)
        
        # Run the test
        result = subprocess.run(
            [sys.executable, '-m', 'pytest', category['path'], '-v', '--tb=short'],
            capture_output=True,
            text=True
        )
        
        if result.returncode == 0:
            print(f"✅ {category['name']}: PASSED")
            # Count passed tests from output
            passed = result.stdout.count(' PASSED')
            total_passed += passed
            print(f"   {passed} tests passed")
        else:
            print(f"❌ {category['name']}: FAILED")
            # Count failed tests from output
            failed = result.stdout.count(' FAILED')
            total_failed += failed
            print(f"   {failed} tests failed")
            print(f"   Error output:\n{result.stdout}")
    
    print(f"\n{'=' * 50}")
    print(f"📊 Test Summary")
    print(f"{'=' * 50}")
    print(f"✅ Total Passed: {total_passed}")
    print(f"❌ Total Failed: {total_failed}")
    print(f"📈 Success Rate: {(total_passed / (total_passed + total_failed)) * 100:.1f}%")
    
    if total_failed == 0:
        print(f"\n🎉 All tests passed! The notification system is ready for production.")
        return 0
    else:
        print(f"\n⚠️  Some tests failed. Please review the failures above.")
        return 1

def demo_functionality():
    """Demonstrate notification system functionality."""
    
    print(f"\n🚀 Notification System Demo")
    print("=" * 50)
    
    try:
        # Import and test basic functionality
        from ethoscope_node.notifications.base import NotificationAnalyzer
        from ethoscope_node.notifications.email import EmailNotificationService
        
        print("✅ Successfully imported notification classes")
        
        # Test initialization
        analyzer = NotificationAnalyzer()
        email_service = EmailNotificationService()
        
        print("✅ Successfully created notification instances")
        
        # Test basic functionality
        test_device_id = "DEMO_DEVICE_001"
        
        # Test failure analysis (will return error for non-existent device)
        analysis = analyzer.analyze_device_failure(test_device_id)
        print(f"✅ Device failure analysis: {analysis.get('error', 'Success')}")
        
        # Test duration formatting
        duration_str = analyzer._format_duration(3665)  # 1 hour, 1 minute, 5 seconds
        print(f"✅ Duration formatting: {duration_str}")
        
        # Test email configuration test
        config_test = email_service.test_email_configuration()
        print(f"✅ Email configuration test: {config_test.get('success', 'Tested')}")
        
        print(f"\n🎯 Key Features Demonstrated:")
        print("   • Device failure analysis with detailed context")
        print("   • Experiment duration calculation and formatting")
        print("   • User email resolution and admin notifications")
        print("   • Log file attachment for debugging")
        print("   • Rate limiting and cooldown mechanisms")
        print("   • Multi-protocol SMTP support (TLS/SSL)")
        print("   • Comprehensive error handling")
        
        return True
        
    except Exception as e:
        print(f"❌ Demo failed: {e}")
        return False

if __name__ == "__main__":
    print("🔧 Ethoscope Notification System Test Suite")
    print("=" * 60)
    
    # Run functionality demo
    demo_success = demo_functionality()
    
    if demo_success:
        # Run comprehensive tests
        test_result = run_tests()
        sys.exit(test_result)
    else:
        print("❌ Demo failed - skipping tests")
        sys.exit(1)