"""
Unit tests for database cache and discovery functionality.

This module tests the database listing and caching features used for
the append functionality dropdown population.
"""

import pytest
import unittest
import tempfile
import os
import sqlite3
import json
import time
from unittest.mock import Mock, patch, MagicMock

# Import the classes we're testing
from ethoscope.io import (
    get_all_databases_info,
    _fallback_database_discovery,
    SQLiteDatabaseMetadataCache,
    MySQLDatabaseMetadataCache,
    create_metadata_cache
)


@pytest.fixture
def temp_cache_dir():
    """Create a temporary cache directory for testing."""
    temp_dir = tempfile.mkdtemp(prefix="ethoscope_cache_test_")
    yield temp_dir
    
    # Cleanup
    import shutil
    if os.path.exists(temp_dir):
        shutil.rmtree(temp_dir)


@pytest.fixture
def sample_cache_files(temp_cache_dir):
    """Create sample cache files for testing."""
    device_name = "ETHOSCOPE_001"
    
    # Create SQLite cache file
    sqlite_cache_data = {
        "db_name": f"2023-01-15_10-30-00_{device_name}.db",
        "device_name": device_name,
        "tracking_start_time": "2023-01-15_10-30-00",
        "creation_timestamp": 1673775000.0,
        "db_status": "finalised",
        "db_size_bytes": 1024000,
        "table_counts": {"ROI_1": 1500, "ROI_2": 1450},
        "last_db_update": 1673775000.0,
        "db_version": "SQLite 3.39.0",
        "experiment_info": {
            "date_time": 1673775000.0,
            "backup_filename": f"2023-01-15_10-30-00_{device_name}.db",
            "user": "researcher",
            "location": "lab_A",
            "result_writer_type": "SQLiteResultWriter",
            "sqlite_source_path": f"/ethoscope_data/results/{device_name}/2023-01-15_10-30-00_{device_name}.db",
            "run_id": "exp_001"
        }
    }
    
    sqlite_cache_file = os.path.join(temp_cache_dir, f"db_metadata_2023-01-15_10-30-00_{device_name}_db.json")
    with open(sqlite_cache_file, 'w') as f:
        json.dump(sqlite_cache_data, f, indent=2)
    
    # Create MySQL cache file
    mysql_cache_data = {
        "db_name": f"{device_name}_db",
        "device_name": device_name,
        "tracking_start_time": "2023-01-16_14-20-00",
        "creation_timestamp": 1673875200.0,
        "db_status": "finalised",
        "db_size_bytes": 2048000,
        "table_counts": {"ROI_1": 2500, "ROI_2": 2450, "ROI_3": 2300},
        "last_db_update": 1673875200.0,
        "db_version": "MySQL 8.0.31",
        "experiment_info": {
            "date_time": 1673875200.0,
            "backup_filename": f"2023-01-16_14-20-00_{device_name}.db",
            "user": "researcher",
            "location": "lab_B",
            "result_writer_type": "MySQLResultWriter",
            "run_id": "exp_002"
        }
    }
    
    mysql_cache_file = os.path.join(temp_cache_dir, f"db_metadata_2023-01-16_14-20-00_{device_name}_db.json")
    with open(mysql_cache_file, 'w') as f:
        json.dump(mysql_cache_data, f, indent=2)
    
    return {
        "sqlite_file": sqlite_cache_file,
        "mysql_file": mysql_cache_file,
        "device_name": device_name
    }


@pytest.fixture
def temp_sqlite_databases(temp_cache_dir):
    """Create temporary SQLite databases for fallback discovery testing."""
    device_name = "ETHOSCOPE_002"
    results_dir = os.path.join(temp_cache_dir, "results", device_name)
    os.makedirs(results_dir, exist_ok=True)
    
    databases = []
    
    # Create a few test SQLite databases
    for i, timestamp in enumerate(["2023-01-10_09-15-00", "2023-01-11_10-30-00"]):
        db_path = os.path.join(results_dir, f"{timestamp}_{device_name}.db")
        
        conn = sqlite3.connect(db_path)
        cursor = conn.cursor()
        
        # Create basic ROI tables
        cursor.execute("""
            CREATE TABLE ROI_1 (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                t INTEGER,
                x REAL,
                y REAL
            )
        """)
        
        cursor.execute(f"INSERT INTO ROI_1 (t, x, y) VALUES ({(i+1)*1000}, 10.5, 20.5)")
        
        conn.commit()
        conn.close()
        
        databases.append(db_path)
    
    return {
        "databases": databases,
        "device_name": device_name,
        "results_dir": results_dir
    }


class TestGetAllDatabasesInfo:
    """Test the get_all_databases_info function."""
    
    def test_get_all_databases_info_success(self, sample_cache_files, temp_cache_dir):
        """Test successful retrieval of database info from cache files."""
        device_name = sample_cache_files["device_name"]
        
        databases_info = get_all_databases_info(device_name, temp_cache_dir)
        
        # Verify structure
        assert isinstance(databases_info, dict)
        assert "SQLite" in databases_info
        assert "MariaDB" in databases_info
        
        # Verify SQLite database info
        sqlite_dbs = databases_info["SQLite"]
        assert len(sqlite_dbs) == 1
        
        sqlite_key = list(sqlite_dbs.keys())[0]
        sqlite_info = sqlite_dbs[sqlite_key]
        
        assert sqlite_info["filesize"] == 1024000
        assert sqlite_info["backup_filename"].endswith(f"{device_name}.db")
        assert sqlite_info["version"] == "SQLite 3.39.0"
        assert sqlite_info["db_status"] == "finalised"
        
        # Verify MySQL database info
        mysql_dbs = databases_info["MariaDB"]
        assert len(mysql_dbs) == 1
        
        mysql_key = list(mysql_dbs.keys())[0]
        mysql_info = mysql_dbs[mysql_key]
        
        assert mysql_info["db_size_bytes"] == 2048000
        assert mysql_info["backup_filename"].endswith(f"{device_name}.db")
        assert mysql_info["version"] == "MySQL 8.0.31"
        assert mysql_info["db_status"] == "finalised"
    
    def test_get_all_databases_info_empty_device_name(self, temp_cache_dir):
        """Test handling of empty device name."""
        databases_info = get_all_databases_info("", temp_cache_dir)
        
        assert databases_info == {"SQLite": {}, "MariaDB": {}}
    
    def test_get_all_databases_info_nonexistent_cache_dir(self):
        """Test handling of non-existent cache directory."""
        databases_info = get_all_databases_info("ETHOSCOPE_999", "/nonexistent/path")
        
        # Should create the directory and return empty results
        assert databases_info == {"SQLite": {}, "MariaDB": {}}
    
    def test_get_all_databases_info_corrupted_cache_file(self, temp_cache_dir):
        """Test handling of corrupted cache files."""
        device_name = "ETHOSCOPE_003"
        
        # Create a corrupted cache file
        corrupted_file = os.path.join(temp_cache_dir, f"db_metadata_2023-01-17_12-00-00_{device_name}_db.json")
        with open(corrupted_file, 'w') as f:
            f.write("{ invalid json content }")
        
        databases_info = get_all_databases_info(device_name, temp_cache_dir)
        
        # Should handle the error gracefully and return empty results
        assert databases_info == {"SQLite": {}, "MariaDB": {}}
    
    def test_get_all_databases_info_missing_experiment_info(self, temp_cache_dir):
        """Test handling of cache files missing experiment_info."""
        device_name = "ETHOSCOPE_004"
        
        # Create cache file without experiment_info
        cache_data = {
            "db_name": f"{device_name}_db",
            "device_name": device_name,
            "db_size_bytes": 500000,
            "table_counts": {"ROI_1": 100}
            # Missing experiment_info section
        }
        
        cache_file = os.path.join(temp_cache_dir, f"db_metadata_2023-01-18_15-00-00_{device_name}_db.json")
        with open(cache_file, 'w') as f:
            json.dump(cache_data, f)
        
        databases_info = get_all_databases_info(device_name, temp_cache_dir)
        
        # Should skip files without proper experiment_info
        assert databases_info == {"SQLite": {}, "MariaDB": {}}


class TestFallbackDatabaseDiscovery:
    """Test the fallback database discovery functionality."""
    
    def test_fallback_discovery_success(self, temp_sqlite_databases):
        """Test successful fallback database discovery."""
        device_name = temp_sqlite_databases["device_name"]
        cache_dir = os.path.dirname(temp_sqlite_databases["results_dir"])
        
        # Mock the search paths to include our test directory structure
        with patch('ethoscope.utils.cache._fallback_database_discovery') as mock_fallback:
            # Call the real function
            mock_fallback.side_effect = lambda dn, cd: _fallback_database_discovery(dn, cd)
            
            databases = _fallback_database_discovery(device_name, cache_dir)
            
            # Verify SQLite databases were discovered
            assert "SQLite" in databases  
            sqlite_dbs = databases["SQLite"]
            
            # Should find our test databases
            assert len(sqlite_dbs) >= 1  # At least one database should be found
            
            # Verify database properties
            for db_name, db_info in sqlite_dbs.items():
                assert db_info["file_exists"] is True
                assert db_info["version"] == "SQLite 3.x"
                assert db_info["db_status"] == "discovered"
                assert db_info["filesize"] > 0
    
    def test_fallback_discovery_no_databases_found(self, temp_cache_dir):
        """Test fallback discovery when no databases are found."""
        device_name = "NONEXISTENT_DEVICE"
        
        databases = _fallback_database_discovery(device_name, temp_cache_dir)
        
        assert databases == {"SQLite": {}, "MariaDB": {}}
    
    def test_fallback_discovery_permission_error(self, temp_cache_dir):
        """Test fallback discovery handling permission errors."""
        device_name = "ETHOSCOPE_005"
        
        # Create a directory we can't read
        restricted_dir = os.path.join(temp_cache_dir, "restricted")
        os.makedirs(restricted_dir, exist_ok=True)
        
        # This should not crash even if we can't access some directories
        databases = _fallback_database_discovery(device_name, temp_cache_dir)
        
        # Should return empty results without crashing
        assert isinstance(databases, dict)
        assert "SQLite" in databases
        assert "MariaDB" in databases


class TestSQLiteDatabaseMetadataCache:
    """Test SQLite database metadata cache functionality."""
    
    def test_create_sqlite_cache(self):
        """Test creation of SQLite metadata cache."""
        db_credentials = {"name": "/tmp/test.db"}
        device_name = "ETHOSCOPE_TEST"
        cache_dir = "/tmp/cache"
        
        cache = create_metadata_cache(
            db_credentials=db_credentials,
            device_name=device_name,
            cache_dir=cache_dir,
            database_type="SQLite3"
        )
        
        assert isinstance(cache, SQLiteDatabaseMetadataCache)
        assert cache.db_credentials == db_credentials
        assert cache.device_name == device_name
    
    def test_create_mysql_cache(self):
        """Test creation of MySQL metadata cache."""
        db_credentials = {
            "name": "test_db",
            "user": "user",
            "password": "pass"
        }
        device_name = "ETHOSCOPE_TEST"
        cache_dir = "/tmp/cache"
        
        cache = create_metadata_cache(
            db_credentials=db_credentials,
            device_name=device_name,
            cache_dir=cache_dir,
            database_type="MySQL"
        )
        
        assert isinstance(cache, MySQLDatabaseMetadataCache)
        assert cache.db_credentials == db_credentials
        assert cache.device_name == device_name
    
    def test_auto_detect_database_type(self):
        """Test automatic database type detection."""
        # SQLite detection
        sqlite_credentials = {"name": "/path/to/database.db"}
        sqlite_cache = create_metadata_cache(
            db_credentials=sqlite_credentials,
            device_name="TEST"
        )
        assert isinstance(sqlite_cache, SQLiteDatabaseMetadataCache)
        
        # MySQL detection
        mysql_credentials = {"name": "mysql_database"}
        mysql_cache = create_metadata_cache(
            db_credentials=mysql_credentials,
            device_name="TEST"
        )
        assert isinstance(mysql_cache, MySQLDatabaseMetadataCache)


class TestDatabaseListIntegration:
    """Integration tests for database list functionality used in tracking."""
    
    def test_database_list_creation(self, sample_cache_files, temp_cache_dir):
        """Test creation of database list for frontend dropdown."""
        device_name = sample_cache_files["device_name"]
        
        databases_info = get_all_databases_info(device_name, temp_cache_dir)
        
        # Simulate the database list creation from tracking.py
        db_list = []
        if databases_info and databases_info.get("SQLite"):
            db_list.extend(databases_info["SQLite"].keys())
        if databases_info and databases_info.get("MariaDB"):
            db_list.extend(databases_info["MariaDB"].keys())
        
        # Verify the list contains databases from both types
        assert len(db_list) == 2  # One SQLite + One MySQL
        assert any(db.endswith('.db') for db in db_list)  # SQLite database name
    
    def test_empty_database_list_handling(self, temp_cache_dir):
        """Test handling when no databases are found."""
        device_name = "EMPTY_DEVICE"
        
        databases_info = get_all_databases_info(device_name, temp_cache_dir)
        
        # Simulate the database list creation
        db_list = []
        if databases_info and databases_info.get("SQLite"):
            db_list.extend(databases_info["SQLite"].keys())
        if databases_info and databases_info.get("MariaDB"):
            db_list.extend(databases_info["MariaDB"].keys())
        
        # Should handle empty list gracefully
        assert db_list == []
    
    def test_fallback_when_cache_empty(self, temp_sqlite_databases):
        """Test that fallback discovery is triggered when cache is empty."""
        device_name = temp_sqlite_databases["device_name"]
        cache_dir = os.path.dirname(temp_sqlite_databases["results_dir"])
        
        # Call get_all_databases_info with empty cache
        databases_info = get_all_databases_info(device_name, cache_dir)
        
        # Should trigger fallback and find SQLite databases
        if databases_info["SQLite"]:  # If fallback worked
            assert len(databases_info["SQLite"]) > 0
            
            # Create database list
            db_list = list(databases_info["SQLite"].keys())
            assert len(db_list) > 0
            assert all(db.endswith('.db') for db in db_list)


if __name__ == '__main__':
    # Configure logging for tests
    import logging
    logging.basicConfig(level=logging.DEBUG)
    
    # Run the tests
    pytest.main([__file__, '-v'])